
/**********************************************************************
 * 
 * lrt_power.c
 *
 * copyright (c) 2006, Wei Sun, UCLA
 *
 * last modified Oct 20, 2006
 * first written Oct 20, 2006
 *
 * Licensed under the GNU General Public License version 2 (June, 1991)
 *
 * C functions for the R/DBC package
 *
 **********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <R.h>
#include <Rmath.h>
#include "ttest.h"

/**********************************************************************
 * 
 * print_v
 *
 * print out a vector of type double
 *
 **********************************************************************/

void Rprint_v(double* v, int nrl, int nrh)
{
	int i;
	for (i = nrl; i < nrh; i++){
		Rprintf ("%f\t", v[i]);
	}
	Rprintf ("%f\n", v[i]);
}

void Rprint_ve(double* v, int nrl, int nrh)
{
	int i;
	for (i = nrl; i < nrh; i++){
		Rprintf ("%.2e\t", v[i]);
	}
	Rprintf ("%.2e\n", v[i]);
}

void Rprint_vi(int* v, int nrl, int nrh)
{
	int i;
	for (i = nrl; i < nrh; i++){
		Rprintf ("%d\t", v[i]);
	}
	Rprintf ("%d\n", v[i]);
}


/**********************************************************************
 * 
 * reorg
 *
 * Reorganize a vector to a matrix of given size. 
 *
 * Allocation done by R_alloc, so that R does the cleanup.
 *
 **********************************************************************/

void reorg(double *v, double ***m, int nrow, int ncol)
{
    int i;
    
    *m = (double **)R_alloc(nrow, sizeof(double*));
    
    (*m)[0] = v;
    if(nrow>1){
        for(i=1; i<nrow; i++){
            (*m)[i] = (*m)[i-1] + ncol;
        }
    }
}

/**********************************************************************
 * 
 * coef
 *
 * coefficients of linear model lm(y ~ x)
 *
 **********************************************************************/

void coef(double* y, double* x, int n, double *a, double *b)
{
    int i;
    double ave_x = 0;
    double ave_y = 0;
    double sum_xy = 0.0;
    double sum_xx = 0.0;
    
    for(i=0; i<n; i++){
        ave_x += x[i];
        ave_y += y[i];
    }
    
    ave_x /= n;
    ave_y /= n;
    
    for(i=0; i<n; i++){
        sum_xy += x[i]*y[i];
        sum_xx += x[i]*x[i];
    }
    sum_xy -= n*ave_x*ave_y;
    sum_xx -= n*ave_x*ave_x;
    
    *b = sum_xy/sum_xx;
    *a = ave_y - (*b) * ave_x;
}

/**********************************************************************
 * 
 * lrt_haploid
 *
 *
 * likelihood ratio test between non-nested models
 *    yd = a0 + b0 * x0
 *    yd = a1 + b1 * x1
 *
 **********************************************************************/

void lrt_haploid(double *y, double *x0, double *x1, int *nR, double *pval){

    int i;
    int n = *nR;
    double a0, b0, a1, b1, lsigma01, diff, sigma0, sigma1;
    double *r0, *r1, lr01, tmp0, tmp1, w01, z1;

	r0 = (double *) calloc(n, sizeof(double));
	r1 = (double *) calloc(n, sizeof(double));

    coef(y, x0, n, &a0, &b0);
    coef(y, x1, n, &a1, &b1);
    
    for(i=0; i<n; i++){
        r0[i] = y[i] - a0 - b0*x0[i];
        r0[i] = r0[i]*r0[i];
        r1[i] = y[i] - a1 - b1*x1[i];
        r1[i] = r1[i]*r1[i];
    }
    
    sigma0 = 0.0;
    sigma1 = 0.0;
    
    for(i=0; i<n; i++){
        sigma0 += r0[i];
        sigma1 += r1[i];
    }
    
    sigma0 /= n;
    sigma0  = sqrt(sigma0);
    
    sigma1 /= n;
    sigma1  = sqrt(sigma1);
    
    tmp0 = 2*sigma0*sigma0;
    tmp1 = 2*sigma1*sigma1;
    
    lsigma01 = log(sigma0) - log(sigma1);
    lr01 = -n*lsigma01;
    
    w01 = 0.0;
    for(i=0; i<n; i++){
        diff = r0[i]/tmp0  - r1[i]/tmp1 + lsigma01;
        w01 += (diff*diff);
    }
    
    w01 /= n;
    w01 -= (lsigma01*lsigma01);
    w01 *= n;
    
    z1  = lr01/sqrt(w01);
    
    /*
     * double pnorm(double x, double mu, double sigma, int lower_tail,
     *            int give_log);
     */
    
    *pval = pnorm(z1, 0.0, 1.0, 0, 0);
    
    free(r0);
    free(r1);
}

/**********************************************************************
 * 
 * lrt_powr
 *
 * given genotypes of three markers: x1, x2, x3, generate y
 * so that x1 is the true QTL and x2, x3 are pseudo QTL
 *
 * Calculate the LRT test p-value between x1, x2 and between x2, x3
 *
 **********************************************************************/

void lrt_power(int *dims, double *x1, double *x2, double *x3, double* sigmaR, double* pvR, double *pjR){

    double *y, **pj, t, p1, p2, p3, plrt;
    int n_sample = dims[0]; // total number of yeasts
    int n = dims[1]; // number of genes linked to one marker
    int d = dims[2]; // number of mismatched genotype
    int ns2 = n_sample/2;
    int j, k;

    reorg(pjR, &pj, n, 2);
    
	y = (double *) calloc(n_sample+1, sizeof(double));
    
    GetRNGstate();

    j = 0;
    while(j < n){
        for(k=0; k<n_sample; k++){
            y[k+1] = x1[k] + norm_rand()*sigmaR[j];
        }
        
        /**
         * NOTICE: tutest is FUNCTION FROM NUMERICAL RECIPE, SO IT TAKE 
         * y[1:n2] AS ONE VECTOR AND y[(n2+1):n] AS THE OTHER VECTOR
         */
        tutest(y, ns2, &y[ns2], ns2, &t, &p1);
                
        if(p1 > 2*pvR[j] || p1 < 0.5*pvR[j]){ continue; }
        
        tutest(y, ns2-d, &y[ns2-d], ns2+d, &t, &p2);
        if(p2 < p1){ continue; }

        tutest(y, ns2+d, &y[ns2+d], ns2-d, &t, &p3);
        if(p3 < p1){ continue; }

        lrt_haploid(&y[1], x1, x2, &n_sample, &plrt);
        pj[j][0] = plrt;
        
        if(p2 < p3){
            lrt_haploid(&y[1], x2, x3, &n_sample, &plrt);
        }else{
            lrt_haploid(&y[1], x3, x2, &n_sample, &plrt);
        }
        pj[j][1] = plrt;
        
        j++;
    }
    
    PutRNGstate();
    
    free(y);
}
