
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>

static double dsqrarg;
#define DSQR(a) ((dsqrarg=(a)) == 0.0 ? 0.0 : dsqrarg*dsqrarg)


void nrerror(char error_text[])
/* Numerical Recipes standard error handler */
{
	fprintf(stderr,"Numerical Recipes run-time error...\n");
	fprintf(stderr,"%s\n",error_text);
	fprintf(stderr,"...now exiting to system...\n");
	exit(1);
}


/**********************************************************************
 * 
 * betacf
 *
 * continued fraction used by betai
 * 
 **********************************************************************/

#define MAXIT 100
#define EPS 3.0e-7
#define FPMIN 1.0e-30

double betacf(double a, double b, double x)
{
	int m,m2;
	double aa,c,d,del,h,qab,qam,qap;

	qab=a+b;
	qap=a+1.0;
	qam=a-1.0;
	c=1.0;
	d=1.0-qab*x/qap;
	if (fabs(d) < FPMIN) d=FPMIN;
	d=1.0/d;
	h=d;
	for (m=1;m<=MAXIT;m++) {
		m2=2*m;
		aa=m*(b-m)*x/((qam+m2)*(a+m2));
		d=1.0+aa*d;
		if (fabs(d) < FPMIN) d=FPMIN;
		c=1.0+aa/c;
		if (fabs(c) < FPMIN) c=FPMIN;
		d=1.0/d;
		h *= d*c;
		aa = -(a+m)*(qab+m)*x/((a+m2)*(qap+m2));
		d=1.0+aa*d;
		if (fabs(d) < FPMIN) d=FPMIN;
		c=1.0+aa/c;
		if (fabs(c) < FPMIN) c=FPMIN;
		d=1.0/d;
		del=d*c;
		h *= del;
		if (fabs(del-1.0) < EPS) break;
	}
	if (m > MAXIT) nrerror("a or b too big, or MAXIT too small in betacf");
	return h;
}
#undef MAXIT
#undef EPS
#undef FPMIN

/**********************************************************************
 * 
 * gammln
 *
 * logarithm of gamma function 
 * 
 **********************************************************************/

double gammln(double xx)
{
	double x,y,tmp,ser;
	static double cof[6]={76.18009172947146,-86.50532032941677,
		24.01409824083091,-1.231739572450155,
		0.1208650973866179e-2,-0.5395239384953e-5};
	int j;

	y=x=xx;
	tmp=x+5.5;
	tmp -= (x+0.5)*log(tmp);
	ser=1.000000000190015;
	for (j=0;j<=5;j++) ser += cof[j]/++y;
	return -tmp+log(2.5066282746310005*ser/x);
}

/**********************************************************************
 * 
 * betai
 *
 * incomplete beta function 
 * 
 **********************************************************************/

double betai(double a, double b, double x)
{
	double betacf(double a, double b, double x);
	double gammln(double xx);
	double bt;

	if (x < 0.0 || x > 1.0) nrerror("Bad x in routine betai");
	if (x == 0.0 || x == 1.0) bt=0.0;
	else
		bt=exp(gammln(a+b)-gammln(a)-gammln(b)+a*log(x)+b*log(1.0-x));
	if (x < (a+1.0)/(a+b+2.0))
		return bt*betacf(a,b,x)/a;
	else
		return 1.0-bt*betacf(b,a,1.0-x)/b;
}

/**********************************************************************
 * 
 * avevar
 *
 * compute the mean and variance 
 * 
 **********************************************************************/
void avevar(double *data, int n, double *ave, double *var)
{
	int j;
	double s,ep;

	for (*ave=0.0,j=1;j<=n;j++) *ave += data[j];
	*ave /= n;
	*var=ep=0.0;
	for (j=1;j<=n;j++) {
		s=data[j]-(*ave);
		ep += s;
		*var += s*s;
	}
	*var=(*var-ep*ep/n)/(n-1);
}

/**********************************************************************
 * 
 * ttest
 * 
 * t-test with combined variance 
 *
 **********************************************************************/
void ttest(double *data1, int n1, double *data2, int n2, double *t, double *prob)
{
	void avevar(double data[], int n, double *ave, double *var);
	double betai(double a, double b, double x);
	double var1,var2,svar,df,ave1,ave2;

	avevar(data1,n1,&ave1,&var1);
	avevar(data2,n2,&ave2,&var2);
	df=n1+n2-2;
	svar=((n1-1)*var1+(n2-1)*var2)/df;
	*t=(ave1-ave2)/sqrt(svar*(1.0/n1+1.0/n2));
	*prob=betai(0.5*df,0.5,df/(df+(*t)*(*t)));
}

/**********************************************************************
 * 
 * ttest
 * 
 * t-test with un-equal variance 
 *
 **********************************************************************/
void tutest(double *data1, int n1, double *data2, int n2, double *t, double *prob)
{
	void avevar(double data[], int n, double *ave, double *var);
	double betai(double a, double b, double x);
	double var1,var2,df,ave1,ave2;

	avevar(data1,n1,&ave1,&var1);
	avevar(data2,n2,&ave2,&var2);
	printf("ave1=%f, ave2=%f, var1=%f, var2=%f\n", ave1, ave2, var1, var2);
	*t=(ave1-ave2)/sqrt(var1/n1+var2/n2);
	df=DSQR(var1/n1+var2/n2)/(DSQR(var1/n1)/(n1-1)+DSQR(var2/n2)/(n2-1));
	printf("df=%f\n", df);
	*prob=betai(0.5*df,0.5,df/(df+DSQR(*t)));
}

/**********************************************************************
 * 
 * coef
 *
 * coefficients of linear model lm(y ~ x)
 *
 **********************************************************************/

void coef(double* y, double* x, int n, double *a, double *b)
{
    int i;
    double ave_x = 0;
    double ave_y = 0;
    double sum_xy = 0.0;
    double sum_xx = 0.0;
    
    for(i=0; i<n; i++){
        ave_x += x[i];
        ave_y += y[i];
    }
    
    ave_x /= n;
    ave_y /= n;
    
    for(i=0; i<n; i++){
        sum_xy += x[i]*y[i];
        sum_xx += x[i]*x[i];
    }
    sum_xy -= n*ave_x*ave_y;
    sum_xx -= n*ave_x*ave_x;
    
    *b = sum_xy/sum_xx;
    *a = ave_y - (*b) * ave_x;
}



int main(){

double x[57] = {0, 3.58e+00, 2.23e+00, -1.65e-01, 1.37e+00, -1.14e+00, 1.48e+00, -1.37e+00, 2.41e+00, 1.66e+00, -2.79e-03, 2.90e+00, -1.50e+00, -3.35e-01, 2.83e+00, -1.16e+00, -1.34e-01, -6.14e-01, -4.81e-01, -7.20e-01, -8.56e-01, 1.81e+00, -3.26e-01, 7.05e-01, -5.47e-01, 7.04e-01, -2.65e+00, -2.18e+00, -2.90e-01, 3.79e+00, -5.96e-01, 7.40e-01, 7.80e-01, -1.22e+00, -2.37e+00, -1.35e-02, -9.12e-01, -1.10e+00, 1.45e+00, -1.62e+00, 8.94e-02, -4.83e-01, 5.79e-01, -1.11e-01, 1.91e+00, -7.16e-01, -1.80e-01, -1.01e+00, 1.72e+00, -1.91e+00, 1.76e-01, 8.00e-01, -6.10e-01, 4.37e-01, -1.20e+00, 2.94e-01, 7.53e-01};

double y[57] = {0, -5.84e-01, 1.88e+00, 2.43e+00, 4.62e-01, -5.55e-01, 5.40e-01, 1.17e+00, 1.32e+00, 2.36e+00, 9.68e-01, -2.68e-01, 2.74e+00, -8.40e-01, 2.10e+00, 2.90e+00, 4.35e-02, 3.13e-01, 2.28e+00, 9.72e-01, 1.72e+00, 2.47e+00, 8.35e-01, 6.22e-01, 1.61e+00, 5.59e-01, 1.46e+00, 2.22e+00, 3.07e+00, 7.51e-01, 2.17e+00, -3.53e+00, 1.25e+00, -1.92e-01, 2.31e+00, 1.41e+00, -8.83e-01, 2.63e+00, 1.29e+00, 2.45e+00, -1.22e+00, 7.14e-01, 8.65e-01, -7.99e-01, 1.58e+00, 4.62e-01, -2.38e-01, 2.88e-02, 1.32e-01, 1.23e+00, -1.20e+00, 7.27e-01, 5.04e-01, 1.49e-01, 1.86e+00, -2.13e-02, -1.68e+00};

double t, prob, a, b;
    
tutest(x, 56, y, 56, &t, &prob);

printf("t=%f, p=%e\n", t, prob);

coef(y, x, 57, &a, &b);
printf("a=%f, b=%f\n", a, b);

}

